<?php

namespace App\Http\Controllers;

use App\Models\Club;
use App\Models\Athlete;
use App\Models\Country;
use App\Models\Category;
use App\Models\AgeCategory;
use App\Models\Level;
use Illuminate\Http\Request;
use App\Services\AthleteImportService;
class AthleteController extends Controller
{
    // Show all athletes
    public function index()
    {
        $user = auth()->user();

        if ($user && $user->is_club) {
            // Load only this club's athletes; handle missing club gracefully
            $club = Club::where('user_id', $user->id)->first();
            $athletes = $club ? Athlete::where('club_id', $club->id)->get() : collect();
        } else {
            $athletes = Athlete::all();
        }

        return view('athletes.index', compact('athletes'));
    }

    // Show create form
    public function create()
    {
        $user = auth()->user();

        if ($user && $user->is_club) {
            // Only the logged-in club
            $club = Club::where('user_id', $user->id)->first();
            $clubs = $club ? collect([$club]) : collect();
        } else {
            $clubs = Club::all();
        }

        $countries = Country::all();
        $ageCategories = AgeCategory::all();
        $levels = Level::all();
        $categories = Category::all();

        return view('athletes.create', compact('clubs', 'countries', 'ageCategories', 'levels', 'categories'));
    }

    // Store new athlete
    public function store(Request $request)
    {
        $request->validate([
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'gender' => 'required|in:male,female',
            'dob' => 'required|date',
            'club_id' => 'required|exists:clubs,id',
            'country_id' => 'required|exists:countries,id',
            'age_category_id' => 'required|exists:age_categories,id',
            'level_id' => 'required|exists:levels,id',
            'category_id' => 'required|exists:categories,id',
            'photo_url' => 'nullable|image', // ✅ image validation
        ]);

        // Prepare data
        $data = $request->only(['first_name', 'last_name', 'gender', 'dob', 'club_id', 'country_id', 'age_category_id', 'level_id', 'type', 'category_id']);

        // ✅ Handle image upload
        if ($request->hasFile('photo_url')) {
            $file = $request->file('photo_url');
            $filename = time() . '_' . $file->getClientOriginalName();
            $file->move(public_path('uploads/athletes'), $filename); // stored in public/uploads/athletes
            $data['photo_url'] = 'uploads/athletes/' . $filename; // save relative path
        }

        Athlete::create($data);

        return redirect()->route('athletes.index')->with('success', 'Athlete created successfully.');
    }

    // Show edit form
    public function edit($id)
    {
        $athlete = Athlete::findOrFail($id);

        $user = auth()->user();

        if ($user && $user->is_club) {
            // Only the logged-in club
            $club = Club::where('user_id', $user->id)->first();
            $clubs = $club ? collect([$club]) : collect();
        } else {
            $clubs = Club::all();
        }

        $countries = Country::all();
        $ageCategories = AgeCategory::all();
        $levels = Level::all();
        $categories = Category::all();

        return view('athletes.edit', compact('athlete', 'clubs', 'countries', 'ageCategories', 'levels', 'categories'));
    }

    // Update athlete
    public function update(Request $request, $id)
    {
        $request->validate([
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'gender' => 'required|in:male,female',
            'dob' => 'required|date',
            'club_id' => 'required|exists:clubs,id',
            'country_id' => 'required|exists:countries,id',
            'age_category_id' => 'required|exists:age_categories,id',
            'level_id' => 'required|exists:levels,id',
            'category_id' => 'required|exists:categories,id',
            'photo_url' => 'nullable|image', // image optional
        ]);

        $athlete = \App\Models\Athlete::findOrFail($id);

        // Collect updated fields
        $data = $request->only(['first_name', 'last_name', 'gender', 'dob', 'club_id', 'type', 'country_id', 'age_category_id', 'level_id', 'category_id']);

        // ✅ Handle new photo upload
        if ($request->hasFile('photo_url')) {
            // Delete old photo if exists
            if ($athlete->photo_url && file_exists(public_path($athlete->photo_url))) {
                unlink(public_path($athlete->photo_url));
            }

            // Save new one
            $file = $request->file('photo_url');
            $filename = time() . '_' . $file->getClientOriginalName();
            $file->move(public_path('uploads/athletes'), $filename);

            $data['photo_url'] = 'uploads/athletes/' . $filename;
        }

        $athlete->update($data);

        return redirect()->route('athletes.index')->with('success', 'Athlete updated successfully.');
    }

    // Delete athlete
    public function destroy($id)
    {
        $athlete = Athlete::findOrFail($id);
        $athlete->delete();

        return redirect()->route('athletes.index')->with('success', 'Athlete deleted successfully.');
    }

    public function import(Request $request, AthleteImportService $importService)
    {
        $request->validate([
            'file' => 'required|file|mimes:csv,txt,xlsx',
        ]);

        $result = $importService->import($request->file('file'));

        return back()
            ->with('success', "{$result['success_count']} athletes imported successfully.")
            ->with('import_errors', $result['failed_rows']);
    }
}
