<?php

namespace App\Http\Controllers\Club;

use App\Http\Controllers\Controller;
use App\Models\AgeCategory;
use App\Models\Athlete;
use App\Models\Competition;
use App\Models\CompetitionAthlete;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class CompetitionAthleteController extends Controller
{
    /**
     * Screen 1: AgeCategory list (no competition yet)
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        abort_unless($user && !empty($user->is_club) && $user->is_club, 403, 'Club access required.');

        $ageCategories = AgeCategory::query()->orderBy('name')->get();

        return view('ownerclub.registration.index', compact('ageCategories'));
    }

    /**
     * Screen 2: Eligible athletes for chosen AgeCategory + Competition dropdown
     */
    public function athletes(AgeCategory $ageCategory, Request $request)
    {
        $user = Auth::user();
        abort_unless($user && $user->is_club, 403, 'Club access required.');

        // --- Start query ---
        $eligible = Athlete::query()
            ->where('club_id', $user->club->id ?? null)
            ->where('age_category_id', $ageCategory->id);

        $min = $ageCategory->min_age;
        $max = $ageCategory->max_age;

        // 🟢 Case 1: Full date-based category (e.g., 2018-01-01 → 2018-12-31)
        // if ($min && preg_match('/^\d{4}-\d{2}-\d{2}$/', $min)) {
        //     $eligible->whereDate('dob', '>=', $min)
        //              ->whereDate('dob', '<=', $max);
        // }
        // // 🟢 Case 2: Year-only range (e.g., min_age = 2018, max_age = 2019)
        // elseif ($min && preg_match('/^\d{4}$/', $min)) {
        //     $eligible->whereYear('dob', '>=', (int)$min)
        //              ->whereYear('dob', '<=', (int)$max);
        // }
        // // 🟢 Case 3: Numeric ages (e.g., 10–12 years)
        // else {
        //     $today = Carbon::today();
        //     if (!is_null($min)) {
        //         $dobMax = $today->copy()->subYears((int)$min)->endOfDay();
        //         $eligible->whereDate('dob', '<=', $dobMax);
        //     }
        //     if (!is_null($max)) {
        //         $dobMin = $today->copy()->subYears((int)$max)->startOfDay();
        //         $eligible->whereDate('dob', '>=', $dobMin);
        //     }
        // }

        // --- Search filter ---
        if ($request->filled('search')) {
            $s = trim($request->get('search'));
            $eligible->where(function ($q) use ($s) {
                $q->where('first_name', 'like', "%{$s}%")->orWhere('last_name', 'like', "%{$s}%");
            });
        }

        // --- Get final results ---
        $eligibleAthletes = $eligible->orderBy('first_name')->orderBy('last_name')->get();

        // --- Active competitions (datetime-aware) ---
        $now = Carbon::now();
        $competitions = Competition::orderBy('name')
            ->get()
            ->map(function ($c) use ($now) {
                $start = $c->start_date ? Carbon::parse($c->start_date) : null;
                $end = $c->end_date ? Carbon::parse($c->end_date) : null;
                $isOpen = $start && $end ? $now->between($start, $end) : true;

                return [
                    'id' => $c->id,
                    'name' => $c->name,
                    'is_open' => $isOpen,
                    'window' => $start && $end ? $start->format('Y-m-d H:i') . ' → ' . $end->format('Y-m-d H:i') : 'No window set',
                ];
            });

        $alreadyIds = [];

        return view('ownerclub.registration.athletes', [
            'ageCategory' => $ageCategory,
            'eligibleAthletes' => $eligibleAthletes,
            'competitions' => $competitions,
            'alreadyIds' => $alreadyIds,
            'search' => $request->get('search'),
        ]);
    }

    /**
     * Store: competition_id comes from form
     */
    public function store(AgeCategory $ageCategory, Request $request)
    {
        $user = Auth::user();
        abort_unless($user && !empty($user->is_club) && $user->is_club, 403, 'Club access required.');

        $data = $request->validate([
            'competition_id' => 'required|exists:competitions,id',
            'athlete_ids' => 'array',
            'athlete_ids.*' => 'exists:athletes,id',
        ]);

        $competition = Competition::findOrFail($data['competition_id']);

        // Active window check (using competition dates)
        $now = Carbon::now();
        $start = $competition->start_date ? Carbon::parse($competition->start_date) : null;
        $end = $competition->end_date ? Carbon::parse($competition->end_date) : null;
        $isOpen = $start && $end ? $now->between($start, $end) : true;

        if (!$isOpen) {
            return back()->with('error', 'Registration window is closed for the selected competition.');
        }

        // Only this club’s athletes
        $incoming = collect($data['athlete_ids'] ?? [])
            ->unique()
            ->filter(function ($id) use ($user) {
                return Athlete::where('id', $id)->where('club_id', $user?->club?->id)->exists();
            })
            ->values();

        DB::transaction(function () use ($incoming, $competition, $ageCategory, $user) {
            // Current for THIS competition + THIS age category + THIS club
            $current = CompetitionAthlete::query()->where('competition_id', $competition->id)->whereHas('athlete', fn($q) => $q->where('club_id', $user?->club?->id))->pluck('athlete_id');

            $toAdd = $incoming->diff($current);
            $toRemove = $current->diff($incoming);

            foreach ($toAdd as $athleteId) {
                CompetitionAthlete::create([
                    'competition_id' => $competition->id,
                    'athlete_id' => $athleteId,
                    'group_id' => null, // groups assigned later
                ]);
            }

            if ($toRemove->isNotEmpty()) {
                CompetitionAthlete::where('competition_id', $competition->id)->whereIn('athlete_id', $toRemove)->delete();
            }
        });

        return redirect()
            ->route('club.registration.athletes', [$ageCategory->id])
            ->with('success', 'Selections saved for the chosen competition.');
    }

    public function export(AgeCategory $ageCategory, Request $request)
    {
         $user = Auth::user();
        abort_unless($user && $user->is_club, 403, 'Club access required.');

        $competitionId = $request->query('competition_id');
        if (!$competitionId) {
            return back()->with('error', 'Competition ID is required.');
        }

        $competition = Competition::find($competitionId);
        if (!$competition) {
            return back()->with('error', 'Competition not found.');
        }

        // 🟢 Find athletes of this club and age category who are registered in this competition
        $athleteIds = CompetitionAthlete::where('competition_id', $competitionId)
            ->whereHas('athlete', function ($q) use ($user, $ageCategory) {
                $q->where('club_id', $user->club->id)->where('age_category_id', $ageCategory->id);
            })
            ->pluck('athlete_id')
            ->toArray();

        if (empty($athleteIds)) {
            return back()->with('error', 'No registered athletes found for this category.');
        }

        $athletes = Athlete::whereIn('id', $athleteIds)
            ->with(['country:id,name', 'level:id,name', 'category:id,name'])
            ->orderBy('first_name')
            ->get();

        // Split into chunks (max 40 athletes per sheet)
        $chunks = $athletes->chunk(40);

        // 🧾 Build Excel export dynamically
        $export = new class ($chunks, $competition, $ageCategory) implements \Maatwebsite\Excel\Concerns\WithMultipleSheets {
            protected $chunks;
            protected $competition;
            protected $ageCategory;

            public function __construct($chunks, $competition, $ageCategory)
            {
                $this->chunks = $chunks;
                $this->competition = $competition;
                $this->ageCategory = $ageCategory;
            }

            public function sheets(): array
            {
                $sheets = [];
                foreach ($this->chunks as $i => $chunk) {
                    $sheets[] = new class ($chunk, $i + 1, $this->competition, $this->ageCategory) implements \Maatwebsite\Excel\Concerns\FromCollection, \Maatwebsite\Excel\Concerns\WithTitle, \Maatwebsite\Excel\Concerns\WithHeadings {
                        protected $chunk;
                        protected $index;
                        protected $competition;
                        protected $ageCategory;

                        public function __construct($chunk, $index, $competition, $ageCategory)
                        {
                            $this->chunk = $chunk;
                            $this->index = $index;
                            $this->competition = $competition;
                            $this->ageCategory = $ageCategory;
                        }

                        public function collection()
                        {
                            return $this->chunk->map(function ($a, $i) {
                                return [
                                    '#' => $i + 1,
                                    'First Name' => $a->first_name,
                                    'Last Name' => $a->last_name,
                                    'DOB' => $a->dob ? Carbon::parse($a->dob)->format('Y-m-d') : '',
                                    'Country' => $a->country->name ?? '',
                                    'Level' => $a->level->name ?? '',
                                    'Category' => $a->category->name ?? '',
                                    'Gender' => ucfirst($a->gender),
                                ];
                            });
                        }

                        public function headings(): array
                        {
                            return ['#', 'First Name', 'Last Name', 'DOB', 'Country', 'Level', 'Category', 'Gender'];
                        }

                        public function title(): string
                        {
                            return 'Sheet ' . $this->index;
                        }
                    };
                }

                return $sheets;
            }
        };

        $fileName = sprintf('%s_%s_%s.xlsx', str_replace(' ', '_', $competition->name), str_replace(' ', '_', $ageCategory->name), now()->format('Ymd_His'));

        return Excel::download($export, $fileName);
    }
}
