<?php

namespace App\Http\Controllers;

use App\Models\Club;
use App\Models\Country;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;

class ClubController extends Controller
{
    public function index()
    {
        // Include user so you can show login email in the list if needed
        $clubs = Club::with(['country', 'user'])
            ->latest()
            ->get();
        return view('clubs.index', compact('clubs'));
    }

    public function create()
    {
        $countries = Country::all();
        return view('clubs.create', compact('countries'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'country_id' => ['required', 'exists:countries,id'],
            // Allow precise datetime input (HTML datetime-local format)
            'start_date' => ['nullable', 'date_format:Y-m-d\TH:i'],
            'end_date' => ['nullable', 'date_format:Y-m-d\TH:i', 'after_or_equal:start_date'],

            // Club login credentials
            'email' => ['required', 'email', 'unique:users,email'],
            'password' => ['required', 'confirmed', 'min:8'],
        ]);

        return DB::transaction(function () use ($validated) {
            // 1) Create the user account for the club
            $user = User::create([
                'first_name' => $validated['name'],
                'last_name' => $validated['name'],
                'email' => $validated['email'],
                'password' => $validated['password'], // auto-hashed by cast
                'profile' => 'club',
                'is_club' => true,
                'is_judge' => false,
                'is_admin' => false,
                'is_main_judge' => false,
            ]);

            // 2) Create the club entry
            Club::create([
                'name' => $validated['name'],
                'country_id' => $validated['country_id'],
                'user_id' => $user->id,
                // Laravel will auto-cast datetime strings into Carbon instances
                'start_date' => $validated['start_date'],
                'end_date' => $validated['end_date'],
            ]);

            return redirect()->route('clubs.index')->with('success', 'Club created successfully.');
        });
    }
    public function edit($id)
    {
        $club = Club::with('user')->findOrFail($id);
        $countries = Country::all();
        return view('clubs.edit', compact('club', 'countries'));
    }

    public function update(Request $request, $id)
    {
        $club = Club::with('user')->findOrFail($id);

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'country_id' => ['required', 'exists:countries,id'],
            'start_date' => ['nullable', 'date_format:Y-m-d\TH:i'],
            'end_date' => ['nullable', 'date_format:Y-m-d\TH:i', 'after_or_equal:start_date'],

            // Email + optional password update
            'email' => ['required', 'email', Rule::unique('users', 'email')->ignore(optional($club->user)->id)],
            'password' => ['nullable', 'confirmed', 'min:8'],
        ]);

        return DB::transaction(function () use ($club, $validated) {
            // 1) Update or create linked user
            if ($club->user) {
                $club->user->email = $validated['email'];
                if (!empty($validated['password'])) {
                    $club->user->password = $validated['password']; // auto-hashed
                }
                $club->user->is_club = true;
                $club->user->profile = $club->user->profile ?: 'club';
                $club->user->save();
            } else {
                $user = User::create([
                    'first_name' => $validated['name'],
                    'last_name' => null,
                    'email' => $validated['email'],
                    'password' => $validated['password'] ?? str()->random(16),
                    'profile' => 'club',
                    'is_club' => true,
                    'is_judge' => false,
                    'is_admin' => false,
                    'is_main_judge' => false,
                ]);
                $club->user_id = $user->id;
            }

            // 2) Update club details
            $club->update([
                'name' => $validated['name'],
                'country_id' => $validated['country_id'],
                'start_date' => $validated['start_date'],
                'end_date' => $validated['end_date'],
            ]);

            return redirect()->route('clubs.index')->with('success', 'Club updated successfully.');
        });
    }

    public function destroy($id)
    {
        $club = \App\Models\Club::with('user')->findOrFail($id);

        return \DB::transaction(function () use ($club) {
            // 1) Ensure a placeholder club exists
            $placeholder = \App\Models\Club::firstOrCreate(
                ['name' => 'Independent'],
                ['location' => '-', 'email' => null], // fill required columns
            );

            // 2) Reassign athletes to the placeholder
            \App\Models\Athlete::where('club_id', $club->id)->update(['club_id' => $placeholder->id]);

            // 3) Delete linked user (optional) and the club
            if ($club->user) {
                $club->user->delete(); // or soft delete
            }

            $club->delete();

            return redirect()->route('clubs.index')->with('success', 'Club deleted. Athletes reassigned to Independent.');
        });
    }

    public function getAthletes(\App\Models\Club $club)
    {
        $athletes = $club->athletes()->select('id', 'first_name', 'last_name')->orderBy('last_name')->get();

        return response()->json($athletes);
    }
}
