<?php

namespace App\Http\Controllers;

use App\Models\Competition;
use Illuminate\Http\Request;
use Carbon\Carbon;
use App\Models\Club;
use App\Models\Group;
use App\Models\Level;
use App\Models\Score;
use App\Models\Athlete;
use App\Models\Apparatus;
use App\Models\AgeCategory;
use App\Models\RoundRotation;
use App\Models\CompetitionAthlete;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
class CompetitionController extends Controller
{
    // Show all competitions
    public function index()
    {
        $user = Auth::user();

        if ($user->is_judge) {
            // Only show competitions assigned to this judge via apparatus assignment
            $competitions = Competition::whereHas('assignedJudges', function ($query) use ($user) {
                $query->where('user_id', $user->id);
            })
                ->latest()
                ->get();

            return view('judge.competitions.index', compact('competitions'));
        } else {
            // Show all competitions for admins
            $competitions = Competition::latest()->get();

            return view('competitions.index', compact('competitions'));
        }
    }

    // Show create form
    public function create()
    {
        return view('competitions.create');
    }

    // Store new competition
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'location' => 'required|string|max:255',
            'scoring_type' => 'required|string',
            'gender' => 'required|in:male,female',
        ]);

        Competition::create([
            'name' => $request->name,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'location' => $request->location,
            'scoring_type' => $request->scoring_type,
            'gender' => $request->gender,
            'enable_vault2' => $request->has('enable_vault2'),
        ]);

        return redirect()->route('competitions.index')->with('success', 'Competition created successfully.');
    }

    // Show edit form
    public function edit($id)
    {
        $competition = Competition::findOrFail($id);
        return view('competitions.edit', compact('competition'));
    }

    // Update competition
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'location' => 'required|string|max:255',
            'scoring_type' => 'required|string',
            'gender' => 'required|in:male,female',
        ]);

        $competition = Competition::findOrFail($id);

        $competition->update([
            'name' => $request->name,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'location' => $request->location,
            'scoring_type' => $request->scoring_type,
            'gender' => $request->gender,
            // ✅ update Vault 2 toggle
            'enable_vault2' => $request->has('enable_vault2'),
        ]);

        return redirect()->route('competitions.index')->with('success', 'Competition updated successfully.');
    }

    // Delete competition
    public function destroy($id)
    {
        $competition = Competition::findOrFail($id);

        // 1. Delete athletes assigned directly to the competition (FK: competition_id)
        $competition->competitionAthletes()->delete();

        // 2. Delete athletes assigned to each group (FK: group_id)
        foreach ($competition->groups as $group) {
            $group->competitionAthletes()->delete();
        }

        // 3. Delete all groups of this competition
        $competition->groups()->delete();

        // 4. Now delete the competition itself
        $competition->delete();

        return redirect()->route('competitions.index')->with('success', 'Competition and all related data deleted successfully.');
    }

    public function generateRotations($competitionId)
    {
        RoundRotation::where('competition_id', $competitionId)->delete();
        $competition = Competition::findOrFail($competitionId);
        $groups = Group::where('competition_id', $competitionId)->orderBy('id')->get();
        $apparatuses = Apparatus::orderBy('id')->take(4)->get();

        if ($groups->count() !== 4 && $groups->count() !== 8) {
            return redirect()->back()->with('success', '4/8 groups and 4/8 apparatuses are required.');
        }
        DB::transaction(function () use ($competitionId, $groups, $apparatuses) {
            for ($round = 0; $round < 4; $round++) {
                foreach ($groups as $index => $group) {
                    $apparatusIndex = ($index + $round) % 4;
                    RoundRotation::create([
                        'competition_id' => $competitionId,
                        'round_number' => $round + 1,
                        'group_id' => $group->id,
                        'apparatus_id' => $apparatuses[$apparatusIndex]->id,
                    ]);
                }
            }
        });

        return redirect()->back()->with('success', 'Rotations generated successfully!');
    }

  public function showReport(Request $request, $id)
{
    $competition = Competition::findOrFail($id);

    $apparatusList = Apparatus::pluck('name', 'id');

    // ============================================================
    // 1. GET GLOBAL TOP-3 (FULL COMPETITION, IGNORE FILTER)
    // ============================================================
    $allAthletesForCompetition = CompetitionAthlete::with([
        'athlete.scores' => function ($q) use ($id) {
            $q->where('competition_id', $id);
        }
    ])->where('competition_id', $id)->get();

    $globalTop3 = $allAthletesForCompetition->map(function ($item) use ($apparatusList) {

        $scoresByApp = $item->athlete->scores->keyBy('apparatus_id');

        $total = 0;

        foreach ($apparatusList as $appid => $name) {
            $total += $scoresByApp[$appid]->final_score ?? 0;
        }

        return [
            'athlete_id' => $item->athlete_id,
            'final' => $total
        ];

    })->sortByDesc('final')
      ->take(3)
      ->pluck('athlete_id')
      ->toArray();

    // ============================================================
    // 2. APPLY USER FILTERS ONLY FOR DISPLAY
    // ============================================================
    $levelFilter = (array) $request->level;
    $ageCategoryFilter = (array) $request->age_category;

    $athletesQuery = CompetitionAthlete::with([
        'athlete.club',
        'athlete.ageCategory',
        'athlete.level',
        'athlete.scores' => function ($q) use ($id) {
            $q->where('competition_id', $id);
        },
    ])->where('competition_id', $id);

    if ($request->filled('athlete')) {
        $athletesQuery->where('athlete_id', $request->athlete);
    }
    if ($request->filled('type')) {
        $athletesQuery->whereHas('athlete', function ($q) use ($request) {
            $q->where('type', $request->type);
        });
    }
    if (!empty($levelFilter)) {
        $athletesQuery->whereHas('athlete', function ($q) use ($levelFilter) {
            $q->whereIn('level_id', $levelFilter);
        });
    }
    if (!empty($ageCategoryFilter)) {
        $athletesQuery->whereHas('athlete', function ($q) use ($ageCategoryFilter) {
            $q->whereIn('age_category_id', $ageCategoryFilter);
        });
    }

    // ============================================================
    // 3. BUILD SCORES ONLY FOR FILTERED ATHLETES (DISPLAY ONLY)
    // ============================================================
    $athletes = $athletesQuery->get()->map(function ($item) use ($apparatusList) {
        $scoresByApparatus = $item->athlete->scores->keyBy('apparatus_id');

        $athleteData = [
            'athlete' => $item->athlete,
            'scores' => [],
            'final' => 0,
        ];

        $totalScore = 0;

        foreach ($apparatusList as $id => $name) {
            $score = $scoresByApparatus[$id]->final_score ?? 0;
            $athleteData['scores'][$name] = $score;
            $totalScore += $score;
        }

        $athleteData['final'] = $totalScore;

        return $athleteData;
    })->sortByDesc('final')->values();

    // ============================================================
    // 4. RETURN VIEW
    // ============================================================
    $levels = Level::orderBy('name')->get();
    $ageCategories = AgeCategory::orderBy('name')->get();
    $athleteList = Athlete::orderBy('first_name')->get();
    $clubs = Club::orderBy('name')->get();
    $selectedApparatus = $request->apparatus ?? 'final';

    return view('competitions.report', compact(
        'competition',
        'athletes',
        'levels',
        'ageCategories',
        'apparatusList',
        'athleteList',
        'selectedApparatus',
        'globalTop3'      // << IMPORTANT
    ));
}

    public function createTeam()
    {
        $clubs = \App\Models\Club::orderBy('name')->get();
        return view('competitions.team_create', compact('clubs'));
    }

    public function storeTeam(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'club_a_id' => 'required|different:club_b_id|exists:clubs,id',
            'club_b_id' => 'required|exists:clubs,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'location' => 'required|string|max:255',
            'scoring_type' => 'required|string',
            'club_a_athletes' => 'required|array|min:1',
            'club_b_athletes' => 'required|array|min:1',
        ]);

        if (count($request->club_a_athletes) !== count($request->club_b_athletes)) {
            return back()->withErrors(['team' => 'Both clubs must have the same number of athletes.']);
        }

        return DB::transaction(function () use ($request) {
            // 🏆 Create the Team Competition
            $competition = Competition::create([
                'name' => $request->title,
                'is_team_competition' => true,
                'start_date' => Carbon::parse($request->start_date),
                'end_date' => Carbon::parse($request->end_date),
                'location' => $request->location,
                'scoring_type' => $request->scoring_type,
                'club_a_id' => $request->club_a_id,
                'club_b_id' => $request->club_b_id,
            ]);

            // 🏅 Create two groups (one per club)
            $clubA = Club::find($request->club_a_id);
            $clubB = Club::find($request->club_b_id);

            $groupA = Group::create([
                'competition_id' => $competition->id,
                'name' => $clubA->name . ' (Team A)',
            ]);

            $groupB = Group::create([
                'competition_id' => $competition->id,
                'name' => $clubB->name . ' (Team B)',
            ]);

            // 👥 Assign athletes to their groups
            foreach ($request->club_a_athletes as $id) {
                CompetitionAthlete::create([
                    'competition_id' => $competition->id,
                    'athlete_id' => $id,
                    'group_id' => $groupA->id,
                ]);
            }

            foreach ($request->club_b_athletes as $id) {
                CompetitionAthlete::create([
                    'competition_id' => $competition->id,
                    'athlete_id' => $id,
                    'group_id' => $groupB->id,
                ]);
            }

            return redirect()->route('competitions.show', $competition)->with('success', 'Team competition created successfully! Groups generated automatically.');
        });
    }
}
