<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Score;
use App\Models\Club;
use App\Models\Group;
use App\Models\Level;
use App\Models\Athlete;
use App\Models\Apparatus;
use App\Models\AgeCategory;
use App\Models\Competition;
use App\Models\RoundRotation;
use App\Models\AssignApparatus;
use App\Models\CompetitionAthlete;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Services\Vault2Top8ByGroupsBuilder;
class JudgeCompetitionController extends Controller
{
    // List competitions assigned to this judge
    public function index()
    {
        $user = Auth::user();
        $now = now();

        if ($user->is_main_judge) {
            $competitions = Competition::where('start_date', '<=', $now)->where('end_date', '>=', $now)->get();
        } else {
            $competitions = Competition::whereHas('assignedJudges', function ($query) use ($user) {
                $query->where('user_id', $user->id);
            })
                ->where('start_date', '<=', $now)
                ->where('end_date', '>=', $now)
                ->get();
        }

        return view('judge.competitions.index', compact('competitions'));
    }

    // Set selected competition into session
    public function setCompetition(Request $request)
    {
        $request->validate([
            'competition_id' => 'required|exists:competitions,id',
        ]);

        $competition = Competition::findOrFail($request->competition_id);

        session([
            'competition_id' => $competition->id,
            'competition_name' => $competition->name,
        ]);

        return redirect()->route('judge.round');
    }

    // Entry point for round view
    public function round()
    {
        return redirect()->action([self::class, 'showRound']);
    }

    public function showRound(Request $request)
    {
        $competitionId = session('competition_id');
        if (!$competitionId) {
            return redirect()->route('competitions.index')->with('warning', 'Please select a competition first.');
        }

        $user = Auth::user();

        if ($user->is_main_judge) {
            // Main judge overview
            $roundNumbers = RoundRotation::where('competition_id', $competitionId)->orderBy('round_number')->pluck('round_number')->unique();

            if ($roundNumbers->isEmpty()) {
                return view('judge.round.wait', ['message' => 'No rounds found for this competition.']);
            }

            $rounds = [];
            foreach ($roundNumbers as $roundNumber) {
                $rotations = RoundRotation::where('competition_id', $competitionId)
                    ->where('round_number', $roundNumber)
                    ->with(['apparatus', 'group'])
                    ->get();

                if ($rotations->isNotEmpty()) {
                    $rounds[$roundNumber] = $rotations;
                }
            }

            return view('judge.round.main_view', ['rounds' => $rounds]);
        }

        // For judges: route them to the per-athlete list
        return redirect()->route('judge.round.athletes');
    }

    /**
     * Helper: resolve current active rotation (apparatus + group) for the judge.
     * Returns: [RoundRotation $rotation|null, int|null $roundNumber, int|null $apparatusId]
     */
    protected function resolveActiveRotationForJudge(int $competitionId, $user): array
    {
        // 0) pull competition + ids for Vault/Vault2
        $competition = \App\Models\Competition::findOrFail($competitionId);
        $vault2Id = \App\Models\Apparatus::where('slug', 'vault-2')->orWhere('name', 'Vault 2 (V2)')->value('id');

        // 1) normal assignments
        $assignments = \App\Models\AssignApparatus::where('competition_id', $competitionId)->where('user_id', $user->id)->get();

        if ($assignments->isEmpty()) {
            abort(403, 'You are not assigned to any apparatus in this competition.');
        }

        // 2) try BASE rounds first (your original logic)
        $roundNumbers = \App\Models\RoundRotation::where('competition_id', $competitionId)->orderBy('round_number')->pluck('round_number')->unique();

        foreach ($roundNumbers as $roundNumber) {
            $incomplete = \App\Models\RoundRotation::where('competition_id', $competitionId)
                ->where('round_number', $roundNumber)
                ->where(function ($q) {
                    $q->whereNull('status')->orWhere('status', '!=', 'completed');
                })
                // exclude Vault-2 if you tagged it
                ->where(function ($q) {
                    $q->where('is_vault2', false)->orWhereNull('is_vault2');
                })
                ->exists();

            if (!$incomplete) {
                continue;
            }

            foreach ($assignments as $assignment) {
                $rotations = \App\Models\RoundRotation::where('competition_id', $competitionId)
                    ->where('round_number', $roundNumber)
                    ->where('apparatus_id', $assignment->apparatus_id)
                    ->where(function ($q) {
                        $q->where('is_vault2', false)->orWhereNull('is_vault2');
                    })
                    ->with(['apparatus', 'group'])
                    ->get();

                if ($rotations->isEmpty()) {
                    continue;
                }

                $apparatusJudgeIds = \App\Models\AssignApparatus::where('competition_id', $competitionId)->where('apparatus_id', $assignment->apparatus_id)->pluck('user_id')->toArray();

                sort($apparatusJudgeIds);
                $judgeIndex = array_search($user->id, $apparatusJudgeIds);
                if ($judgeIndex === false) {
                    continue;
                }

                $myRotations = $rotations->values()->filter(function ($rotation, $index) use ($judgeIndex, $apparatusJudgeIds) {
                    return $index % count($apparatusJudgeIds) === $judgeIndex;
                });

                foreach ($myRotations as $rotation) {
                    return [$rotation, $roundNumber, $assignment->apparatus_id];
                }
            }
        }

        // 3) FALLBACK: if base rounds are all completed AND Vault-2 is open, route to V2
        if ($vault2Id && $competition->enable_vault2 && in_array($competition->vault2_status, ['ready', 'open', 'pending'])) {
            // ensure the user has a V2 assignment (A) created earlier
            $v2Assignments = \App\Models\AssignApparatus::where('competition_id', $competitionId)->where('apparatus_id', $vault2Id)->pluck('user_id')->toArray();

            // If for any reason V2 assignments still empty, allow any assigned judge to enter V2
            $judgePool = !empty($v2Assignments) ? $v2Assignments : \App\Models\AssignApparatus::where('competition_id', $competitionId)->pluck('user_id')->toArray();

            sort($judgePool);
            $judgeIndex = array_search($user->id, $judgePool);

            // Pick the first V2 round that isn't completed
            $v2Rounds = \App\Models\RoundRotation::where('competition_id', $competitionId)
                ->where('apparatus_id', $vault2Id)
                ->where(function ($q) {
                    $q->whereNull('status')->orWhere('status', '!=', 'completed');
                })
                ->orderBy('round_number')
                ->with(['apparatus', 'group'])
                ->get();

            if ($v2Rounds->isNotEmpty() && $judgeIndex !== false && count($judgePool) > 0) {
                // Distribute groups among judges like before
                $my = $v2Rounds->values()->filter(function ($rotation, $index) use ($judgeIndex, $judgePool) {
                    return $index % count($judgePool) === $judgeIndex;
                });
                foreach ($my as $rotation) {
                    return [$rotation, $rotation->round_number, $vault2Id];
                }

                // If distribution didn’t land anything (e.g., fewer groups than judges), give the first
                return [$v2Rounds->first(), $v2Rounds->first()->round_number, $vault2Id];
            }
        }

        // 4) nothing left
        return [null, null, null];
    }

    /**
     * NEW: Per-apparatus athlete list with search and status/lock
     */
  public function athleteList(Request $request)
{
    $competitionId = session('competition_id');
    if (!$competitionId) {
        return redirect()->route('competitions.index')
            ->with('warning', 'Please select a competition first.');
    }

    $user = Auth::user();

    [$rotation, $roundNumber, $apparatusId] = $this->resolveActiveRotationForJudge($competitionId, $user);

    if (!$rotation) {
        return view('judge.round.wait', [
            'message' => 'You’ve submitted all your group scores for active rounds. Please wait.',
        ]);
    }

    // Get all athlete IDs in this group
    $athleteIds = CompetitionAthlete::where('competition_id', $competitionId)
        ->where('group_id', $rotation->group_id)
        ->pluck('athlete_id')
        ->toArray();

    // Get all athletes already scored by this judge
    $enteredById = Score::where('competition_id', $competitionId)
        ->where('round_number', $roundNumber)
        ->where('apparatus_id', $apparatusId)
        ->where('group_id', $rotation->group_id)
        ->where('created_by_user_id', $user->id)
        ->pluck('athlete_id')
        ->toArray();

    // ✅ Check: Are all athletes in the group already in enteredById?
    $allEntered = !array_diff($athleteIds, $enteredById); 
    // array_diff returns elements in $athleteIds not found in $enteredById

    if ($allEntered && count($athleteIds) > 0) {
        return view('judge.round.wait', [
            'message' => '✅ All athletes in your group have scores submitted. Please wait for the next rotation.',
        ]);
    }

    // Continue if not all entered
    $query = CompetitionAthlete::where('competition_id', $competitionId)
        ->where('group_id', $rotation->group_id)
        ->with('athlete');

    // Optional search
    $s = $request->get('search');
    if ($s) {
        $query->whereHas('athlete', function ($q) use ($s) {
            $q->where('first_name', 'like', "%$s%")
              ->orWhere('last_name', 'like', "%$s%");
        });
    }

    $entries = $query->orderBy('id')
        ->paginate(12)
        ->appends(['search' => $s]);

    return view('judge.round.athletes', [
        'entries' => $entries,
        'search' => $s,
        'currentRound' => $roundNumber,
        'rotation' => $rotation,
        'apparatusId' => $apparatusId,
        'enteredById' => $enteredById,
    ]);
}


    public function generateRotations($competitionId)
    {
        $competition = Competition::findOrFail($competitionId);
        $groups = Group::where('competition_id', $competitionId)->orderBy('id')->get();
        $apparatuses = Apparatus::where('name', 'like', '%Vault 2%')->get();

        if ($groups->count() !== 4 && $groups->count() !== 8) {
            return redirect()->back()->with('success', '4/8 groups and 4/8 apparatuses are required.');
        }
        DB::transaction(function () use ($competitionId, $groups, $apparatuses) {
            for ($round = 0; $round < 4; $round++) {
                foreach ($groups as $index => $group) {
                    $apparatusIndex = ($index + $round) % 4;
                    RoundRotation::create([
                        'competition_id' => $competitionId,
                        'round_number' => $round + 1,
                        'group_id' => $group->id,
                        'apparatus_id' => $apparatuses[$apparatusIndex]->id,
                    ]);
                }
            }
        });

        return redirect()->back()->with('success', 'Rotations generated successfully!');
    }
    /**
     * NEW: Single athlete scoring form (locks if already submitted)
     */
    public function athleteForm(Request $request, int $athleteId)
    {
        $competitionId = session('competition_id');
        if (!$competitionId) {
            return redirect()->route('competitions.index');
        }

        $user = Auth::user();
        [$rotation, $roundNumber, $apparatusId] = $this->resolveActiveRotationForJudge($competitionId, $user);

        if (!$rotation) {
            return redirect()->route('judge.round.athletes');
        }
        // Already submitted by this judge?
        $exists = Score::where([
            'competition_id' => $competitionId,
            'athlete_id' => $athleteId,
            'apparatus_id' => $apparatusId,
            'round_number' => $roundNumber,
            'group_id' => $rotation->group_id,
            'created_by_user_id' => $user->id,
        ])->exists();

        if ($exists) {
            return redirect()->route('judge.round.athletes')->with('warning', 'This athlete already has a submitted score by you.');
        }

        // Ensure athlete belongs to this rotation group
        $entry = CompetitionAthlete::where('competition_id', $competitionId)->where('group_id', $rotation->group_id)->where('athlete_id', $athleteId)->with('athlete', 'competition')->firstOrFail();

        return view('judge.round.athlete_form', [
            'entry' => $entry,
            'currentRound' => $roundNumber,
            'rotation' => $rotation,
        ]);
    }

    /**
     * NEW: Submit single athlete score, then return to list (athlete locked)
     */
    public function submitAthleteScore(Request $request, int $athleteId)
    {
        $competitionId = session('competition_id');
        if (!$competitionId) {
            return redirect()->route('competitions.index')->with('warning', 'Competition not selected.');
        }

        $user = Auth::user();
        [$rotation, $roundNumber, $apparatusId] = $this->resolveActiveRotationForJudge($competitionId, $user);

        if (!$rotation) {
            return redirect()->route('judge.round.athletes')->with('warning', 'No active rotation.');
        }

        // Ensure athlete belongs to current rotation’s group
        $valid = CompetitionAthlete::where('competition_id', $competitionId)->where('group_id', $rotation->group_id)->where('athlete_id', $athleteId)->exists();

        if (!$valid) {
            return back()->with('warning', 'Athlete not in your current group.');
        }

        // Block double-submission
        $already = Score::where([
            'competition_id' => $competitionId,
            'athlete_id' => $athleteId,
            'apparatus_id' => $apparatusId,
            'round_number' => $roundNumber,
            'group_id' => $rotation->group_id,
            'created_by_user_id' => $user->id,
        ])->exists();

        if ($already) {
            return redirect()->route('judge.round.athletes')->with('warning', 'Already submitted.');
        }

        // Validate inputs
        $data = $request->validate([
            'd_score' => 'nullable|numeric|min:0|max:10',
            'e1_score' => 'nullable|numeric|min:0|max:10',
            'e2_score' => 'nullable|numeric|min:0|max:10',
            'e3_score' => 'nullable|numeric|min:0|max:10',
            'e4_score' => 'nullable|numeric|min:0|max:10',
            'cr_score' => 'nullable|numeric|min:0|max:10',
            'bonus' => 'nullable|numeric|min:0|max:10',
            'nd_score' => 'nullable|numeric|min:0|max:99.99',
        ]);

        // Save score
        $score = new Score([
            'competition_id' => $competitionId,
            'athlete_id' => $athleteId,
            'apparatus_id' => $apparatusId,
            'round_number' => $roundNumber,
            'group_id' => $rotation->group_id,
            'd_score' => $data['d_score'] ?? 0,
            'e1_score' => $data['e1_score'] ?? 0,
            'e2_score' => $data['e2_score'] ?? null,
            'e3_score' => $data['e3_score'] ?? null,
            'e4_score' => $data['e4_score'] ?? null,
            'cr_score' => $data['cr_score'] ?? 0,
            'bonus' => $data['bonus'] ?? 0,
            'nd_score' => min($data['nd_score'] ?? 0, 99.99),
            'created_by_user_id' => $user->id,
        ]);

        $score->final_score = $score->calculateFinalScore();
        $score->save();

        // Mark rotation “active” when judge submits
        if ($rotation->status !== 'completed') {
            $rotation->status = 'active';
            $rotation->save();
        }

        return redirect()->route('judge.round.athletes')->with('success', 'Score submitted. Athlete locked.');
    }

    /**
     * (Legacy bulk) Submit a whole group at once – kept for backward compatibility
     */
    public function submitRound(Request $request)
    {
        $competitionId = session('competition_id');
        $user = Auth::user();

        if (!$competitionId) {
            return redirect()->route('competitions.index')->with('warning', 'Competition not selected.');
        }

        $assignment = AssignApparatus::where('competition_id', $competitionId)->where('user_id', $user->id)->first();

        if (!$assignment) {
            return redirect()->back()->with('warning', 'You are not assigned to this competition.');
        }

        $roundNumber = $request->input('round_number');
        $groupId = $request->input('group_id');
        $apparatusId = $request->input('apparatus_id');

        $athleteIds = CompetitionAthlete::where('competition_id', $competitionId)->where('group_id', $groupId)->pluck('athlete_id');

        foreach ($request->athletes ?? [] as $data) {
            if (!isset($data['athlete_id']) || !in_array($data['athlete_id'], $athleteIds->toArray())) {
                continue;
            }

            $exists = Score::where([
                'competition_id' => $competitionId,
                'athlete_id' => $data['athlete_id'],
                'apparatus_id' => $apparatusId,
                'round_number' => $roundNumber,
                'group_id' => $groupId,
                'created_by_user_id' => $user->id,
            ])->exists();

            if ($exists) {
                continue;
            }

            $score = new Score([
                'competition_id' => $competitionId,
                'athlete_id' => $data['athlete_id'],
                'apparatus_id' => $apparatusId,
                'round_number' => $roundNumber,
                'group_id' => $groupId,
                'd_score' => $data['d_score'] ?? 0,
                'e1_score' => $data['e1_score'] ?? 0,
                'e2_score' => $data['e2_score'] ?? null,
                'e3_score' => $data['e3_score'] ?? null,
                'e4_score' => $data['e4_score'] ?? null,
                'bonus' => $data['bonus'] ?? 0,
                'cr_score' => $data['cr_score'] ?? 0,
                'nd_score' => min($data['nd_score'] ?? 0, 99.99),
                'created_by_user_id' => $user->id,
            ]);

            $score->final_score = $score->calculateFinalScore();
            $score->save();
        }

        $rotation = RoundRotation::where('competition_id', $competitionId)->where('round_number', $roundNumber)->where('apparatus_id', $apparatusId)->where('group_id', $groupId)->first();

        if ($rotation && $rotation->status !== 'completed') {
            $rotation->status = 'active';
            $rotation->save();
        }

        return redirect()->route('judge.round')->with('success', 'Scores submitted successfully.');
    }

    public function completeRound(Request $request, $roundNumber)
    {
        $competitionId = session('competition_id');
        $user = Auth::user();

        if (!$competitionId || !$user->is_main_judge) {
            abort(403);
        }

        $rotations = RoundRotation::where('competition_id', $competitionId)->where('round_number', $roundNumber)->get();

        if ($rotations->isEmpty()) {
            return back()->with('warning', 'No rotations found for this round.');
        }

        // Ensure all are currently marked as "active"
        if ($rotations->contains(fn($r) => $r->status !== 'active')) {
            return back()->with('warning', 'Not all rotations are marked as active yet. Please wait for all judges.');
        }

        // Mark this round completed (your existing behavior)
        RoundRotation::where('competition_id', $competitionId)
            ->where('round_number', $roundNumber)
            ->update(['status' => 'completed']);

        // ---------- NEW: If all base rounds are completed, open Vault 2 ----------
        $competition = Competition::findOrFail($competitionId);

        if ($this->allBaseRoundsCompleted($competitionId)) {
            // If Vault 2 is enabled, prepare the V2 rotations and open the phase
            if ($competition->enable_vault2) {
                // Build one extra V2 rotation per group (idempotent)
                app(Vault2Top8ByGroupsBuilder::class)->build($competition);

                // Move Vault 2 lifecycle forward
                $this->ensureVault2Assignments($competition->id);
                if ($competition->vault2_status === 'pending') {
                    $competition->markVault2Ready();
                }
                if ($competition->vault2_status === 'ready') {
                    $competition->markVault2Open();
                }

                return redirect()
                    ->back()
                    ->with('success', "✅ Round $roundNumber marked completed. Normal compilation finished. Vault 2 is now OPEN.");
            }

            return redirect()
                ->back()
                ->with('success', "✅ Round $roundNumber marked completed. Normal compilation finished.");
        }
        // Base rounds not all completed yet — normal success message
        return redirect()
            ->back()
            ->with('success', "✅ Round $roundNumber has been successfully marked as completed.");
    }

    public function checkActiveRound()
    {
        $competitionId = session('competition_id');
        $user = Auth::user();

        if (!$competitionId) {
            return response()->json(['active' => false]);
        }

        $assignments = AssignApparatus::where('competition_id', $competitionId)->where('user_id', $user->id)->get();

        foreach ($assignments as $assignment) {
            $activeRound = RoundRotation::where('competition_id', $competitionId)->where('status', '!=', 'completed')->where('apparatus_id', $assignment->apparatus_id)->first();

            if ($activeRound) {
                $hasScored = Score::where('competition_id', $competitionId)->where('round_number', $activeRound->round_number)->where('apparatus_id', $assignment->apparatus_id)->where('group_id', $activeRound->group_id)->where('created_by_user_id', $user->id)->exists();

                if (!$hasScored) {
                    return response()->json(['active' => true]);
                }
            }
        }

        return response()->json(['active' => false]);
    }

    public function updateBonusAjax(Request $request, $scoreId)
    {
        $request->validate([
            'bonus' => 'nullable|numeric|min:0',
        ]);

        $score = Score::findOrFail($scoreId);
        $score->bonus = $request->bonus ?? 0;
        $score->save();

        return response()->json(['success' => true, 'message' => 'Bonus updated successfully.']);
    }

    protected function allBaseRoundsCompleted(int $competitionId): bool
    {
        $v2Id = \App\Models\Apparatus::where('slug', 'vault-2')->orWhere('name', 'Vault 2 (V2)')->value('id');

        return !\App\Models\RoundRotation::where('competition_id', $competitionId)
            ->when(
                $v2Id,
                function ($q) use ($v2Id) {
                    // exclude Vault 2 rounds
                    $q->where(function ($qq) use ($v2Id) {
                        $qq->where('is_vault2', false)->orWhereNull('is_vault2')->where('apparatus_id', '!=', $v2Id);
                    });
                },
                function ($q) {
                    // if no V2 id found, rely only on is_vault2 flag
                    $q->where(function ($qq) {
                        $qq->where('is_vault2', false)->orWhereNull('is_vault2');
                    });
                },
            )
            ->where(function ($q) {
                $q->whereNull('status')->orWhere('status', '!=', 'completed');
            })
            ->exists();
    }

    protected function ensureVault2Assignments(int $competitionId): void
    {
        $v1Id = \App\Models\Apparatus::where('slug', 'vault')->orWhere('name', 'Vault (V)')->value('id');
        $v2Id = \App\Models\Apparatus::where('slug', 'vault-2')->orWhere('name', 'Vault 2 (V2)')->value('id');
        if (!$v2Id) {
            return;
        }

        // If no V2 assignments exist, mirror V1 (or mirror all apparatus judges if you prefer)
        $hasV2 = \App\Models\AssignApparatus::where('competition_id', $competitionId)->where('apparatus_id', $v2Id)->exists();
        if ($hasV2) {
            return;
        }

        $source = $v1Id ? \App\Models\AssignApparatus::where('competition_id', $competitionId)->where('apparatus_id', $v1Id)->get() : \App\Models\AssignApparatus::where('competition_id', $competitionId)->get(); // fallback: anyone assigned to anything

        foreach ($source as $row) {
            \App\Models\AssignApparatus::firstOrCreate([
                'competition_id' => $competitionId,
                'apparatus_id' => $v2Id,
                'user_id' => $row->user_id,
            ]);
        }
    }
}
