<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Judge;
use Illuminate\Http\Request;

class JudgeController extends Controller
{
    public function index()
    {
        $judges = User::where('is_judge', true)->latest()->get();
        return view('judges.index', compact('judges'));
    }

    public function create()
    {
        return view('judges.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|confirmed|min:6',
            'nationality' => 'nullable|string|max:100',
            'is_main_judge' => 'nullable|boolean',
        ]);

        $user = User::create([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'email' => $request->email,
            'password' => bcrypt($request->password),
            'is_judge' => true,
            'is_main_judge' => $request->has('is_main_judge'), // checkbox returns true/false
        ]);

        Judge::create([
            'user_id' => $user->id,
            'nationality' => $request->nationality,
        ]);

        return redirect()->route('judges.index')->with('success', 'Judge created successfully.');
    }

    public function edit($id)
    {
        $judge = User::findOrFail($id);
        return view('judges.edit', compact('judge'));
    }

    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $judge = $user->judge;

        $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'nationality' => 'nullable|string|max:100',
            'password' => 'nullable|confirmed|min:6',
            'is_main_judge' => 'nullable|boolean',
        ]);

        $user->update([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'email' => $request->email,
            'password' => $request->password ? bcrypt($request->password) : $user->password,
            'is_main_judge' => $request->has('is_main_judge'),
        ]);

        if ($judge) {
            $judge->update([
                'nationality' => $request->nationality,
            ]);
        }

        return redirect()->route('judges.index')->with('success', 'Judge updated successfully.');
    }

    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $user->judge()?->delete();
        $user->delete();

        return redirect()->route('judges.index')->with('success', 'Judge and associated data deleted successfully.');
    }
}
