<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class ProfileController extends Controller
{
    // Show profile edit form
    public function edit()
    {
        $user = Auth::user();
        return view('profile.edit', compact('user'));
    }

    // Update profile info
public function update(Request $request)
{
    $user = Auth::user();

    $request->validate([
        'first_name' => 'required|string|max:255',
        'last_name' => 'required|string|max:255',
        'email' => 'required|email|unique:users,email,' . $user->id,
        'profile' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
    ]);

    $data = [
        'first_name' => $request->first_name,
        'last_name' => $request->last_name,
        'email' => $request->email,
    ];

    // Handle image upload to public/ directory
    if ($request->hasFile('profile')) {
        $image = $request->file('profile');
        $imageName = 'profile_' . $user->id . '.' . $image->getClientOriginalExtension();
        $image->move(public_path('profiles'), $imageName);
        $data['profile'] = 'profiles/' . $imageName;
    }

    $user->update($data);

    return back()->with('success', 'Profile updated successfully.');
}


    // Show change password form
    public function changePasswordForm()
    {
        return view('profile.password');
    }

    // Handle password update
    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'new_password' => 'required|string|min:8|confirmed',
        ]);

        $user = Auth::user();

        if (!Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'Current password is incorrect.']);
        }

        $user->update([
            'password' => Hash::make($request->new_password),
        ]);

        return back()->with('success', 'Password changed successfully.');
    }
}
