<?php

namespace App\Http\Controllers;

use App\Models\Score;
use Illuminate\Http\Request;

class ScoreController extends Controller
{
    // Show all scores
    public function index()
    {
        $scores = Score::all();
        return view('scores.index', compact('scores'));
    }

    // Show create form
    public function create()
    {
        return view('scores.create');
    }

    // Store new score
    public function store(Request $request)
    {
        $request->validate([
            'athlete_id' => 'required|integer|exists:athletes,id',
            'competition_id' => 'required|integer|exists:competitions,id',
            'value' => 'required|numeric|min:0',
        ]);

        Score::create($request->only(['athlete_id', 'competition_id', 'value']));

        return redirect()->route('scores.index')->with('success', 'Score created successfully.');
    }

    // Show edit form
    public function edit($id)
    {
        $score = Score::findOrFail($id);
        return view('scores.edit', compact('score'));
    }

    // Update score
    public function update(Request $request, $id)
    {
        $request->validate([
            'athlete_id' => 'required|integer|exists:athletes,id',
            'competition_id' => 'required|integer|exists:competitions,id',
            'value' => 'required|numeric|min:0',
        ]);

        $score = Score::findOrFail($id);
        $score->update($request->only(['athlete_id', 'competition_id', 'value']));

        return redirect()->route('scores.index')->with('success', 'Score updated successfully.');
    }

    // Delete score
    public function destroy($id)
    {
        $score = Score::findOrFail($id);
        $score->delete();

        return redirect()->route('scores.index')->with('success', 'Score deleted successfully.');
    }

    public function updateField(Request $request, Score $score)
    {
        $field = $request->input('field');
        $value = $request->input('value');

        // Only allow specific editable fields
        $allowed = ['d_score', 'e1_score', 'e2_score', 'e3_score', 'bonus', 'cr_score', 'nd_score'];

        if (!in_array($field, $allowed)) {
            return response()->json(['success' => false, 'message' => 'Invalid field.']);
        }

        $score->$field = is_numeric($value) ? floatval($value) : null;

        // Optional: update final_score here if needed
        $score->final_score = $score->calculateFinalScore();
        $score->save();

        return response()->json(['success' => true, 'message' => 'Updated successfully.']);
    }

 }
