<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\{Apparatus, Competition, CompetitionAthlete, RoundRotation, Score, Athlete};
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class Vault2Controller extends Controller
{
  public function index() {
    $comps = Competition::todayWithVault2()->orderBy('start_date')->get();
    return view('judge.vault2.index', compact('comps'));
  }

  public function show(Request $req, Competition $competition)
  {
    if ($competition->status !== 'complete') {
      return view('judge.vault2.wait', ['competition'=>$competition, 'message'=>"Compilation not complete. Please wait until it’s finished."]);
    }

    if ($competition->vault2_status==='pending') $competition->markVault2Ready();
    if ($competition->vault2_status==='ready')   $competition->markVault2Open();

    $user = $req->user();
    $v1Id = Apparatus::idVault1(); $v2Id = Apparatus::idVault2();

    // Top-8 IDs from V1
    $top8Ids = Score::where('competition_id',$competition->id)->where('apparatus_id',$v1Id)
      ->selectRaw('athlete_id, MAX(final_score) as v1_total')->groupBy('athlete_id')
      ->orderByDesc('v1_total')->limit(8)->pluck('athlete_id');

    // Find the judge’s active V2 rotation (same group, apparatus = V2)
    $rotation = RoundRotation::where('competition_id',$competition->id)
      ->where('is_vault2', true)
      ->first(); // if you have per-judge mapping, narrow it here

    if (!$rotation) {
      return view('judge.vault2.wait', ['competition'=>$competition, 'message'=>'Vault 2 not started for your panel/group yet.']);
    }

    // Intersect Top-8 with this rotation’s group
    $s = $req->get('search');
    $entries = CompetitionAthlete::where('competition_id',$competition->id)
      ->where('group_id',$rotation->group_id)
      ->whereIn('athlete_id',$top8Ids)
      ->with('athlete')
      ->when($s, fn($q)=>$q->whereHas('athlete', fn($qq)=>$qq
        ->where('first_name','like',"%$s%")->orWhere('last_name','like',"%$s%")->orWhere('club_name','like',"%$s%")))
      ->orderBy('id')->paginate(12)->appends(['search'=>$s]);

    $enteredByJudge = Score::where('competition_id',$competition->id)
      ->where('apparatus_id',$v2Id)
      ->where('created_by_user_id',$user->id)
      ->pluck('athlete_id')->toArray();

    // Panel-wide completion
    $scoredCount = Score::where('competition_id',$competition->id)
      ->where('apparatus_id',$v2Id)
      ->whereIn('athlete_id',$top8Ids)
      ->distinct()->count('athlete_id');

    if ($scoredCount >= $top8Ids->count() && $competition->vault2_status!=='complete') {
      $competition->markVault2Complete();
    }

    return view('judge.vault2.scoring', compact('competition','entries','enteredByJudge','rotation','v2Id'));
  }

  public function store(Request $req, Competition $competition, Athlete $athlete)
  {
    abort_unless($competition->enable_vault2, 409);

    $v2Id = Apparatus::idVault2();

    $data = $req->validate([
      'd_score'  => 'required|numeric|min:0|max:10',
      'e1_score' => 'nullable|numeric|min:0|max:10',
      'e2_score' => 'nullable|numeric|min:0|max:10',
      'e3_score' => 'nullable|numeric|min:0|max:10',
      'e4_score' => 'nullable|numeric|min:0|max:10',
      'bonus'    => 'nullable|numeric|min:0|max:10',
      'cr_score' => 'nullable|numeric|min:0|max:10',
      'nd_score' => 'nullable|numeric|min:0|max:10',
    ]);

    $score = Score::updateOrCreate(
      [
        'competition_id'=>$competition->id,
        'apparatus_id'=>$v2Id,
        'athlete_id'=>$athlete->id,
        'created_by_user_id'=>$req->user()->id,
      ],
      array_merge($data, ['round_number'=>1])
    );

    // ensure final_score
    $score->final_score = $score->calculateFinalScore();
    $score->save();

    return back()->with('success','Vault 2 score saved.');
  }

  // Export JSON stub (hook Excel/PDF later)
  public function export(Competition $competition, $fmt)
  {
    $v1 = Apparatus::idVault1(); $v2 = Apparatus::idVault2();
    $top8 = Score::where('competition_id',$competition->id)->where('apparatus_id',$v1)
      ->selectRaw('athlete_id, MAX(final_score) as v1_total')->groupBy('athlete_id')
      ->orderByDesc('v1_total')->limit(8)->pluck('athlete_id');

    $rows = \App\Models\Athlete::with('club')->whereIn('id',$top8)->get()->map(function($a) use ($competition,$v1,$v2){
      $s1 = Score::where(['competition_id'=>$competition->id,'apparatus_id'=>$v1,'athlete_id'=>$a->id])->orderByDesc('final_score')->first();
      $s2 = Score::where(['competition_id'=>$competition->id,'apparatus_id'=>$v2,'athlete_id'=>$a->id])->orderByDesc('final_score')->first();
      $v1t = $s1->final_score ?? 0; $v2t = $s2->final_score ?? 0;
      return ['Athlete'=>$a->first_name.' '.$a->last_name,'Club'=>$a->club->name ?? '-', 'Vault 1'=>$v1t,'Vault 2'=>$v2t,'Sum'=>round($v1t+$v2t,3)];
    })->sortByDesc('Sum')->values()->all();

    if ($fmt==='json') return response()->json($rows);
    abort(501, 'Wire Excel/PDF export with maatwebsite/excel or dompdf.');
  }
}
