<?php

namespace App\Imports;

use App\Models\Competition;
use App\Models\Athlete;
use App\Models\Group;
use App\Models\CompetitionAthlete;
use Maatwebsite\Excel\Row;
use Maatwebsite\Excel\Concerns\OnEachRow;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class CompetitionAthleteImport implements OnEachRow, WithHeadingRow
{
    public $invalid = [];
    public $insertCount = 0;

    public function onRow(Row $row)
    {
        $data = $row->toArray();

        // Clean headings
        if (!isset($data['competition_name']) || !isset($data['athlete_name']) || !isset($data['group_name'])) {
            $this->invalid[] = [
                'row' => $row->getIndex(),
                'competition_name' => $data['competition_name'] ?? '',
                'athlete_name' => $data['athlete_name'] ?? '',
                'group_name' => $data['group_name'] ?? '',
                'error' => 'Invalid headings or missing cells',
            ];
            return;
        }

        // Normalize Excel values
        $competitionName = trim($data['competition_name']);
        $athleteName = preg_replace('/\s+/', ' ', trim($data['athlete_name']));
        $groupName = trim($data['group_name']);

        // Lookup competition
        $competition = Competition::where('name', $competitionName)->first();

        if (!$competition) {
            $this->invalid[] = [
                'row' => $row->getIndex(),
                'competition_name' => $competitionName,
                'athlete_name' => $athleteName,
                'group_name' => $groupName,
                'error' => 'Competition not found',
            ];
            return;
        }

        // Lookup athlete (safer matching)
        $athlete = Athlete::whereRaw(
            "TRIM(REPLACE(CONCAT(first_name,' ',last_name),'  ','')) = ?",
            [$athleteName]
        )->first();

        if (!$athlete) {
            $this->invalid[] = [
                'row' => $row->getIndex(),
                'competition_name' => $competitionName,
                'athlete_name' => $athleteName,
                'group_name' => $groupName,
                'error' => 'Athlete not found',
            ];
            return;
        }

        // Check duplicate
        $exists = CompetitionAthlete::where('competition_id', $competition->id)
            ->where('athlete_id', $athlete->id)
            ->first();

        if ($exists) {
            $this->invalid[] = [
                'row' => $row->getIndex(),
                'competition_name' => $competitionName,
                'athlete_name' => $athleteName,
                'group_name' => $groupName,
                'error' => 'Duplicate entry (already assigned)',
            ];
            return;
        }

        // Lookup group (optional)
        $group = Group::where('name', $groupName)->first();

        // Insert
        CompetitionAthlete::create([
            'competition_id' => $competition->id,
            'athlete_id' => $athlete->id,
            'group_id' => $group?->id,
        ]);

        $this->insertCount++;
    }
}
