<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Competition extends Model
{
    protected $fillable = ['name', 'start_date', 'end_date', 'location', 'scoring_type', 'gender', 'enable_vault2', 'vault2_status', 'vault2_completed_at'];
    protected $casts = [
        'start_date' => 'datetime',
        'end_date' => 'datetime',
    ];
    public function groups()
    {
        return $this->hasMany(Group::class);
    }
    public function assignedJudges()
    {
        return $this->hasMany(AssignApparatus::class);
    }
    public function scores()
    {
        return $this->hasMany(Score::class);
    }

    public function scopeTodayWithVault2($q)
    {
        return $q->whereDate('start_date', now()->toDateString())->where('enable_vault2', true);
    }

    // Top-8 by Vault-1 final_score (aggregate per athlete)
    public function top8Vault1()
    {
        $vault1Id = Apparatus::idVault1();
        if (!$vault1Id) {
            return collect();
        }
        return Score::select('athlete_id', DB::raw('MAX(final_score) as v1_total'))
            ->where('competition_id', $this->id)
            ->where('apparatus_id', $vault1Id)
            ->groupBy('athlete_id')
            ->orderByDesc('v1_total')
            ->orderByDesc('athlete_id')
            ->with(['athlete:id,first_name,last_name,club_id', 'athlete.club:id,name'])
            ->limit(8)
            ->get();
    }

    // lifecycle
    public function markVault2Ready()
    {
        if ($this->enable_vault2) {
            $this->vault2_status = 'ready';
            $this->save();
        }
    }
    public function markVault2Open()
    {
        if ($this->vault2_status === 'ready') {
            $this->vault2_status = 'open';
            $this->save();
        }
    }
    public function markVault2Complete()
    {
        $this->vault2_status = 'complete';
        $this->vault2_completed_at = now();
        $this->save();
    }

    public function certificateTemplate()
    {
        return $this->hasOne(\App\Models\CertificateTemplate::class);
    }

    public function athletes()
    {
        // via pivot CompetitionAthlete
        return $this->belongsToMany(\App\Models\Athlete::class, 'competition_athlete')
            ->withPivot(['group_id'])
            ->withTimestamps();
    }

    public function clubA()
    {
        return $this->belongsTo(Club::class, 'club_a_id');
    }
    public function clubB()
    {
        return $this->belongsTo(Club::class, 'club_b_id');
    }
    public function isTeamCompetition(): bool
    {
        return (bool) $this->is_team_competition;
    }

    public function competitionAthletes()
{
    return $this->hasMany(CompetitionAthlete::class, 'competition_id');
}
}
