<?php

namespace App\Services;

use App\Models\{
    Apparatus,
    Competition,
    Group,
    RoundRotation,
    Score,
    CompetitionAthlete,
    AssignApparatus
};
use Illuminate\Support\Facades\DB;

class Vault2Top8ByGroupsBuilder
{
    public function build(Competition $competition): array
    {
        if (!$competition->enable_vault2) {
            return ['created_round' => 0, 'top8_ids' => []];
        }

        $vault1Id = Apparatus::idVault1();
        $vault2Id = Apparatus::idVault2();
        if (!$vault1Id || !$vault2Id) {
            return ['created_round' => 0, 'top8_ids' => []];
        }

        // Top-8 by Vault-1 (aggregate by athlete)
        $top8 = Score::where('competition_id', $competition->id)
            ->where('apparatus_id', $vault1Id)
            ->selectRaw('athlete_id, MAX(final_score) as v1_total')
            ->groupBy('athlete_id')
            ->orderByDesc('v1_total')
            ->limit(8)
            ->pluck('athlete_id')
            ->all();

        return DB::transaction(function () use ($competition, $vault1Id, $vault2Id, $top8) {

            // 1) Make (or reuse) the virtual group
            $top8Group = Group::firstOrCreate(
                ['competition_id' => $competition->id, 'name' => 'Vault 2'],
                ['order' => 999] // add 'is_virtual' if you have it
            );

            // 2) Create exactly one V2 rotation at round 5 (idempotent)
            $roundNumber = 5;

            $exists = RoundRotation::where([
                'competition_id' => $competition->id,
                'round_number'   => $roundNumber,
                'group_id'       => $top8Group->id,
                'apparatus_id'   => $vault2Id,
            ])->exists();

            if (!$exists) {
                RoundRotation::create([
                    'competition_id' => $competition->id,
                    'round_number'   => $roundNumber,
                    'group_id'       => $top8Group->id,
                    'apparatus_id'   => $vault2Id,
                    'is_vault2'      => true,
                    'status'         => 'pending', // optional
                ]);
            }

            // 3) Reassign ONLY the Top-8 athletes to this group
            if (!empty($top8)) {
                CompetitionAthlete::where('competition_id', $competition->id)
                    ->whereIn('athlete_id', $top8)
                    ->update(['group_id' => $top8Group->id]);
            }

            $vault1JudgeIds = AssignApparatus::where('competition_id', $competition->id)
                ->where('apparatus_id', $vault1Id)
                ->pluck('user_id')
                ->unique()
                ->values();

            if ($vault1JudgeIds->isEmpty()) {
                $vault1JudgeIds = AssignApparatus::where('competition_id', $competition->id)
                    ->pluck('user_id')
                    ->unique()
                    ->values();
            }

            foreach ($vault1JudgeIds as $uid) {
                AssignApparatus::firstOrCreate([
                    'competition_id' => $competition->id,
                    'apparatus_id'   => $vault2Id,
                    'user_id'        => $uid,
                ]);
            }

            return [
                'created_round' => $exists ? 0 : 1,
                'top8_ids'      => $top8,
                'group_id'      => $top8Group->id,
            ];
        });
    }
}
